﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using Elatec;
using System.IO.Ports;
using Microsoft.Win32;
using System.Globalization;
using System.Threading;

namespace Simple_NFC_Demo
{
    public partial class Form1 : Form
    {
        private byte[] Message;
        private SerialPort TWNPort;
        private bool Run = false;
        private ListBoxLogger Logger;

        #region Form1()
        public Form1()
        {
            InitializeComponent();
            // Clear ComboBox with URI types
            cbURIType.Items.Clear();
            // Initialize URI type list
            TURITypes TURIList = new TURITypes();
            // Initialize ComboBox with URI types
            foreach (TSNEPCMD TD in TURIList)
                cbURIType.Items.Add(TD);
            // Select start index
            cbURIType.SelectedIndex = 1;
            // Header, Message Begin, Message End, Short Record, NFC-Forum Wellknown are set
            byte MessageHeader = 0xD1;
            // Set message type to URI
            byte[] Type = new byte[] { 0x55 };
            // payload consists of URI type and string. Standart URI type is http:// www.
            byte[] Payload = new byte[] { 0x01 };
            // Set URI string elatec-rfid.com
            Payload = AddByteArray(Payload, System.Text.ASCIIEncoding.UTF8.GetBytes("elatec-rfid.com"));
            // Add NDEF header to message array
            Message = AddByte2Array(Message, MessageHeader);
            // Add message type length to message array
            Message = AddByte2Array(Message, Convert.ToByte(Type.Length));
            // Add payload length to message array
            Message = AddByte2Array(Message, Convert.ToByte(Payload.Length));
            // Add message type to message array
            Message = AddByteArray(Message, Type);
            // Add payload to message array
            Message = AddByteArray(Message, Payload);
            // Set default address to text boxes
            txtURIString.Text = "elatec-rfid.com";
            txtCurentString.Text = "elatec-rfid.com";
            // Get selected URI type
            TSNEPCMD TURI = (TSNEPCMD)cbURIType.SelectedItem;
            // Write name of selected URI type in 
            txtCurrentURItype.Text = TURI.Name;
            // Initialize Logger
            Logger = new ListBoxLogger(lbLogger);
            // Initialize TWNPort
            TWNPort = new SerialPort();
            // Set Run true
            Run = true;
            // Initialize thread tSearch
            Thread tSearch = new Thread(threadSearch);
            // Start thread tSearch
            tSearch.Start();
        }// End of Form1

        private void btnChangeMessage_Click(object sender, EventArgs e)
        {
            // Clear message array
            Message = null;
            // Header, Message Begin, Message End, Short Record and NFC-Forum Wellknown are set
            byte MessageHeader = 0xD1;
            // Set type to URI
            byte[] Type = new byte[] { 0x55 };
            // Payload consists of URI type and string
            byte[] Payload = new byte[1];
            // Set URI token from selected URI type
            TSNEPCMD TURI = (TSNEPCMD)cbURIType.SelectedItem;
            // Set URI type in payload
            Payload[0] = Convert.ToByte(TURI.Token);
            // Set URI string from text box txtURIString
            Payload = AddByteArray(Payload, System.Text.ASCIIEncoding.UTF8.GetBytes(txtURIString.Text));
            // Add NDEF Header to mssage array
            Message = AddByte2Array(Message, MessageHeader);
            // Add type length to message array
            Message = AddByte2Array(Message, Convert.ToByte(Type.Length));
            // Add payload length to message array
            Message = AddByte2Array(Message, Convert.ToByte(Payload.Length));
            // Add type to message array
            Message = AddByteArray(Message, Type);
            // Add payload to message array
            Message = AddByteArray(Message, Payload);
            // Copy new message to current text boxes
            txtCurentString.Text = txtURIString.Text;
            txtCurrentURItype.Text = TURI.Name;
            // Do log "Message changed"
            Logger.WriteLine("Message changed");
        }// End of btnChangeMessage_Click

        private void cbReceiveMessage_CheckedChanged(object sender, EventArgs e)
        {
            // If receive message is enable, disable send elements
            cbURIType.Enabled = !cbReceiveMessage.Checked;
            txtURIString.Enabled = !cbReceiveMessage.Checked;
            btnChangeMessage.Enabled = !cbReceiveMessage.Checked;
        }// End of cbReadMessage_CheckedChanged

        private void Form1_FormClosing(object sender, FormClosingEventArgs e)
        {
            // Exit thread tSearch
            Run = false;
        }// End of Form1_FormClosing
        #endregion

        #region NFC Communication


        void DoNFCCommunication()
        {
            // Is there a TAG/NFC device?
            if (!SearchTAG())
                // No.
                return;

            // Yes! Wait 500ms till SNEP status reaches state IDLE (=2)
            DateTime Start = DateTime.Now;
            while (SNEP_GetConnectionState() < 2)
                if (DateTime.Now - Start > new TimeSpan(0, 0, 0, 500))
                    // Timeout occured
                    return;

            // Log that we found a device
            Logger.WriteLine("Device found and connected");
            // Should messages received?
            if (cbReceiveMessage.Checked)
            {
                // Yes, do log receiving message
                Logger.WriteLine("Receiving message...");
                // Receive message, while NFC device is connected, state > 1
                while (SNEP_GetConnectionState() > 1)
                {
                    // Receive message and print in logger listbox
                    PrintData(ReceiveMessage(Logger), 16, Logger);
                }
            }
            else
            {
                // Logger write sending message
                Logger.Write("Sending Message...");
                // Is SendMessage successful?
                if (!SendMessage(Message, Logger))
                    // No, Logger write failed
                    Logger.WriteLine("failed");
                else
                {
                    // Yes, Logger write complete
                    Logger.WriteLine("complete");
                    // Logger write device still connected
                    Logger.WriteLine("Device still connected");
                    // Wait, while device is connected, state > 1
                    while (SNEP_GetConnectionState() > 1)
                    {
                    }
                }
            }
            // Logger write device is disconnected
            Logger.WriteLine("Device disconnected");
        }

        private void threadSearch()
        {
            bool IsInit = false;
            // Polling device loop
            while (Run)
            {
                try
                {
                    // Is there a TWN4?
                    string PortName = GetTWNPortName(0);
                    // Is TWNPort open?
                    if (!IsInit)
                    {
                        if (PortName == "")
                            // There is no TWN4 connected. Do a silent execption.
                            throw new ApplicationException("");

                        // Ensure, that COM port becomes available. Sleep a while.
                        System.Threading.Thread.Sleep(100);
                        // A TWN4 was connected. Initialize it.
                        ConnectTWN4(PortName);
                        // Set search tag types to NFC
                        SetTagTypeNFC();
                        // Initialize SNEP service
                        SNEP_Init();
                        // Do log ErrorMessage
                        Logger.WriteLine("TWN4 connected");
                        IsInit = true;
                    }
                    // TWN4 is initialized
                    if (PortName == "")
                        throw new ApplicationException("TWN4 disconnected");

                    // TWN4 is ready for communication. Try to do some kind of NFC communication.                    
                    DoNFCCommunication();
                }
                catch (Exception ex)
                {
                    // An error occured. Show it.
                    if (ex.Message != "")
                        Logger.WriteLine("Error: " + ex.Message);
                    // Try to close TWNPort. Do not generate further exceptions
                    try
                    {
                        if (IsInit)
                            Logger.WriteLine("Disconnecting from TWN4");
                        IsInit = false;
                        TWNPort.Close();
                    }
                    catch { }
                }
            }
            try
            {
                // Try to close TWNPort
                TWNPort.Close();
            }
            catch { }
        }// End of threadSearch

        private bool SendMessage(byte[] MessageBuffer, ListBoxLogger Logger)
        {
            // Is NDEF length = 0?
            if (MessageBuffer.Length == 0)
                // Yes, return true
                return true;
            // Send SNEP_BeginMessage with the message length
            if (!SNEP_BeginMessage(MessageBuffer.Length))
                // Faild, return false
                return false;
            // Define RemainigMessageSize and initializie with message length
            int RemainingMessageSize = MessageBuffer.Length;
            // Send until MessageOffset is less than message length
            for (int MessageOffset = 0; MessageOffset < MessageBuffer.Length; )
            {
                // Initialize MaxFragSize
                int MaxFragSize;
                // Do while buffer free space is 0 
                do
                {
                    // Is NFC device still connected?
                    if (SNEP_GetConnectionState() < 2)
                    {
                        // Log send message failed
                        Logger.WriteLine("failed");
                        // No, return false
                        return false;
                    }
                    // Get current free buffer space
                    if (!GetFragmentByteCount(0, out MaxFragSize))
                    {
                        // No, log Receiving message failed
                        Logger.WriteLine("failed");
                        // Return false
                        return false;
                    }
                }
                while (MaxFragSize == 0);
                // Initialize FragSize with the current buffer free space
                int FragSize = MaxFragSize;
                // Is FragSize higher than RemainingMessageSize
                if (FragSize > RemainingMessageSize)
                    // Yes, initialize ReminingMessageSize in FragSize
                    FragSize = RemainingMessageSize;
                // Do log
                Logger.Write("TX " + FragSize.ToString() + " Bytes: ");
                // Send message fragment from NDEF message and check sending state
                if (!SNEP_SendMessageFrag(GetSegmentFromByteArray(MessageBuffer, MessageOffset, FragSize)))
                    // Sending failed, return false
                    return false;
                // Increment MessageOffset by FragSize
                MessageOffset += FragSize;
                // Decrement RemainingMessageSize by FragSize
                RemainingMessageSize -= FragSize;
                // Do log
                Logger.WriteLine("OK, " + RemainingMessageSize.ToString() + " Bytes remaining");
            }
            // Sending message successful, return true
            return true;
        }// End of SendMessage

        private byte[] ReceiveMessage(ListBoxLogger Logger)
        {
            // Initialize RemainingMessageSize
            int RemainingMessageSize;
            // Initialize ReceiveFragBuffer
            byte[] ReceiveFragBuffer = null;
            // Initialize ReceivedBuffer
            byte[] ReceivedMessage = null;
            // Initialize ReceivedMessage
            ReceivedMessage = null;
            // Is a new message available?
            if (!TestMessage(out RemainingMessageSize))
                // No, return
                return null;
            // Yes, Logger write new line
            Logger.NewLine();
            // do while RemainingMessageSize is larger 0
            do
            {
                // Initialize MaxFragSize
                int MaxFragSize = 0;
                // do while MaxFragSize is 0
                do
                {
                    // Is NFC device still connected?
                    if (SNEP_GetConnectionState() < 2)
                    {
                        // No, log Receiving message failed
                        Logger.WriteLine("Receiving message failed");
                        // Return false
                        return null;
                    }
                    // Is GetFragmentBytecount successfull?
                    if (!GetFragmentByteCount(1, out MaxFragSize))
                    {
                        // No, log Receiving message failed
                        Logger.WriteLine("Receiving message failed");
                        // Return false
                        return null;
                    }
                }
                while (MaxFragSize == 0);
                // Initialize FragSize with the current buffer free space
                int FragSize = MaxFragSize;
                // Do Log
                Logger.Write("TX " + FragSize.ToString() + " Bytes: ");
                // Receive fragment and check is it successfull?
                if (!ReceiveMessageFragment(Convert.ToUInt16(FragSize), out ReceiveFragBuffer))
                {
                    // No, log Receiving message failed
                    Logger.WriteLine("Receiving message failed");
                    // Return false
                    return null;
                }
                // Yes, add received fragment to ReceivedMessage
                ReceivedMessage = AddByteArray(ReceivedMessage, ReceiveFragBuffer);
                // Decrement RemainingMessageSize by FragSize
                RemainingMessageSize -= FragSize;
                // Do Log
                Logger.WriteLine("OK, " + RemainingMessageSize.ToString() + " Bytes remaining");
            }
            while (RemainingMessageSize > 0);
            // Receiving message successful, return true
            return ReceivedMessage;
        }// End of ReceiveMessage

        #region Print Message Tools
        private bool PrintData(byte[] Data, int WindowSize, ListBoxLogger Writer)
        {
            // This function formats the message and print it to a listbox
            int i = 0;
            if (Data == null)
                return false;
            Writer.Write("[" + i.ToString("x8") + "]  ");
            while (i < Data.Length)
            {
                if (i % WindowSize == 0 && i / WindowSize > 0)
                {
                    Writer.WriteLine("  |" + GetMyASCIIString(Data, i - WindowSize, WindowSize) + "|");
                    Writer.Write("[" + i.ToString("x8") + "]  ");
                }
                Writer.Write(" " + Data[i].ToString("X2"));
                i++;
            }
            if (i % WindowSize == 0)
                Writer.WriteLine("  |" + GetMyASCIIString(Data, i - WindowSize, WindowSize) + "|");
            else
            {
                for (int x = 0; x < WindowSize - (i % WindowSize); x++)
                    Writer.Write("   ");
                Writer.Write("  |" + GetMyASCIIString(Data, i - (i % WindowSize), i % WindowSize));
                for (int x = 0; x < WindowSize - (i % WindowSize); x++)
                    Writer.Write(" ");
                Writer.WriteLine("|");
            }
            return true;
        }// End of PrintData
        private string GetMyASCIIString(byte[] Data, int index, int count)
        {
            // This function replace all non chars by •
            if (index + count > Data.Length)
                return null;
            string buffer = null;
            for (int i = 0; i < count; i++)
            {
                if (Data[index + i] > 0x21 && Data[index + i] < 0x7F)
                    buffer = buffer + System.Text.ASCIIEncoding.UTF8.GetString(Data, index + i, 1);
                else
                    buffer = buffer + "•";
            }
            return buffer;
        }// End of GetMyASCIIString
        #endregion
        #endregion

        #region Simple Protocol
        // **************************************************************************************************
        // A Simple Protocol command starts always with two bytes. The first one reflect the API and the 
        // second one the function number.
        // For example the SearchTag function is include in API RF. The related command is 0x18 0x00 0x10.
        // 0x18: API RF
        // 0x00: SearchTag function
        // 0x10: MaxIDBytes
        // **************************************************************************************************

        #region API RF
        private bool SetTagTypeNFC()
        {
            // Generate request set tag type to NFC
            byte[] request = { 0x05, 0x02, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00 };
            // Define Response
            byte[] response = { 0x00 };
            // DoTXRX and copy answer to buffer
            byte[] buffer = DoTXRX(request);
            // DoTXRX and check plausibility
            if (!CompareArraysSegments(response, 0, buffer, 0, 1))
                // Failed, return false
                return false;
            // Return true
            return true;
        }// End of SetTagType
        private bool SearchTAG()
        {
            // Define search request
            byte[] request = { 0x05, 0x00, 0x10 };
            // Define search response
            byte[] response = { 0x00, 0x01 };
            // DoTXRX and copy answer to buffer
            byte[] buffer = DoTXRX(request);
            // Is first byte, the expected byte?
            if (!CompareArraysSegments(response, 0, buffer, 0, 2))
                // Return false
                return false;
            // Return true
            return true;
        }// End of SearchTAG
        #endregion

        #region API NFCSNEP
        private bool SNEP_BeginMessage(int MSGlen)
        {
            // Define SNEP_BeginMessage request header
            byte[] request = { 0x18, 0x03 };
            // Add message length to request
            request = AddByteArray(request, BitConverter.GetBytes(MSGlen));
            // Define SNEP_BeginMessage response
            byte[] response = { 0x00, 0x01 };
            // DoTXRX and copy answer to buffer
            byte[] buffer = DoTXRX(request);
            // DoTXRX and response plausibility check
            if (!CompareArraysSegments(response, 0, buffer, 0, 2))
                // Failed, return false
                return false;
            // Return true
            return true;
        }// End of SNEP_BeginMessage
        private byte SNEP_GetConnectionState()
        {
            // Define SNEP_GetConnectionState request
            byte[] request = { 0x18, 0x01 };
            // Define SNEP_GetConnectionState response
            byte[] response = { 0x00 };
            // DoTXRX
            byte[] buffer = DoTXRX(request);
            // DoTXRX and response plausibility check
            if (!CompareArraysSegments(response, 0, buffer, 0, 1))
                // Failed, return false
                return 0;
            // Get connection state
            return buffer[1];
        }// End of SNEP_GetConnectionState
        private bool GetFragmentByteCount(byte Direction, out int MaxFragmentSize)
        {
            // Initialize MaxFragmentSize
            MaxFragmentSize = 0;
            // Define GetFragmentByteCount request header
            byte[] request = { 0x18, 0x02 };
            // Add direction to request
            request = AddByte2Array(request, Direction);
            // Define GetFragmentByteCount response header
            byte[] response = { 0x00 };
            // DoTXRX
            byte[] buffer = DoTXRX(request);
            // Response plausibility check
            if (!CompareArraysSegments(response, 0, buffer, 0, 1))
                // Failed, return false
                return false;
            // Convert fragment size to int and save in MaxFragmentSize
            MaxFragmentSize = BitConverter.ToUInt16(GetSegmentFromByteArray(buffer, 1, 2), 0);
            // Return true
            return true;
        }// End of GetFragmentByteCount
        private bool SNEP_Init()
        {
            // Define SNEP_Init request header
            byte[] request = { 0x18, 0x00 };
            // Define SNEP_Init response header
            byte[] response = { 0x00, 0x01 };
            // DoTXRX and copy answer to buffer
            byte[] buffer = DoTXRX(request);
            // Response plausibility check
            if (!CompareArraysSegments(response, 0, buffer, 0, 2))
                // Failed, return false
                return false;
            // SNEP init successful, return true
            return true;
        }// End of SNEP_Init
        private bool ReceiveMessageFragment(UInt16 FragmentLen, out byte[] ReceivedMessage)
        {
            // Initialize ReceivedMessage
            ReceivedMessage = null;
            // Define ReceiveMessageFragment request header
            byte[] request = { 0x18, 0x06 };
            // Add FragmentLen to request
            request = AddByteArray(request, BitConverter.GetBytes(FragmentLen));
            // Define SNEP_Init response header
            byte[] reaponse = { 0x00, 0x01 };
            // DoTXRX and copy answer to buffer
            byte[] buffer = DoTXRX(request);
            // Response plausibility check
            if (!CompareArraysSegments(reaponse, 0, buffer, 0, 2))
                // Failed, return false
                return false;
            // Copy message segment in ReceivedMessage
            ReceivedMessage = GetSegmentFromByteArray(buffer, 4, buffer.Length - 4);
            // Return true
            return true;
        }// End of ReceiveMessageFragment
        private bool SNEP_SendMessageFrag(byte[] Message)
        {
            // Define SNEP_SendMessageFrag request header
            byte[] request = { 0x18, 0x04 };
            // Add message length to request
            request = AddByteArray(request, BitConverter.GetBytes(Convert.ToUInt16(Message.Length)));
            // Add message to request
            request = AddByteArray(request, Message);
            // Define SNEP_SendMessageFrag response
            byte[] respnonse = { 0x00, 0x01 };
            // DoTXRX and copy answer to buffer
            byte[] buffer = DoTXRX(request);
            // Response plausibility check
            if (!CompareArraysSegments(respnonse, 0, buffer, 0, 2))
                // Failed, return false
                return false;
            // Return, true
            return true;
        }// End of SNEP_SendMessageFrag
        private bool TestMessage(out int MessageLength)
        {
            // Initialize MessageLength
            MessageLength = 0;
            // Define TestMessage request header
            byte[] request = { 0x18, 0x05 };
            // Define TestMessage response
            byte[] reaponse = { 0x00, 0x01 };
            // DoTXRX and copy answer to buffer
            byte[] buffer = DoTXRX(request);
            // Response plausibility check
            if (!CompareArraysSegments(reaponse, 0, buffer, 0, 2))
                // Failed, return false
                return false;
            // Convert fragment size to int and save in MaxFragmentSize
            MessageLength = BitConverter.ToUInt16(GetSegmentFromByteArray(buffer, 2, 4), 0);
            // Return, true
            return true;
        }// End of TestMessage
        #endregion

        #region Tools for Simple Protocol
        private byte[] GetByteArrayfromPRS(string PRSString)
        {
            // Is string length = 0?
            if (PRSString.Length < 1)
                // Yes, return null
                return null;
            // Initialize byte array, half string length
            byte[] buffer = new byte[PRSString.Length / 2];
            // Get byte array from PRS string
            for (int i = 0; i < (buffer.Length); i++)
                // Convert PRS Chars to byte array buffer
                buffer[i] = byte.Parse(PRSString.Substring((i * 2), 2), NumberStyles.HexNumber);
            // Return byte array
            return buffer;
        }// End of PRStoByteArray
        private string GetPRSfromByteArray(byte[] PRSStream)
        {
            // Is length of PRS stream = 0
            if (PRSStream.Length < 1)
                // Yes, return null
                return null;
            // Iinitialize PRS buffer
            string buffer = null;
            // Convert byte to PRS string
            for (int i = 0; i < PRSStream.Length; i++)
                // convert byte to characters
                buffer = buffer + PRSStream[i].ToString("X2");
            // return buffer
            return buffer;
        }// End of GetPRSfromByteArray
        #endregion
        #endregion

        #region Reader Communication

        private void ConnectTWN4(string PortName)
        {
            // Initialize serial port
            TWNPort.PortName = PortName;
            TWNPort.BaudRate = 115200;
            TWNPort.DataBits = 8;
            TWNPort.StopBits = System.IO.Ports.StopBits.One;
            TWNPort.Parity = System.IO.Ports.Parity.None;
            // NFC functions are known to take less than 2 second to execute.
            TWNPort.ReadTimeout = 2000;
            TWNPort.WriteTimeout = 2000;
            TWNPort.NewLine = "\r";
            // Open TWN4 com port
            TWNPort.Open();
        }// End of ConnectTWN4
        private byte[] DoTXRX(byte[] CMD)
        {
            // Discard com port inbuffer
            TWNPort.DiscardInBuffer();
            // Generate simple protocol string and send command
            TWNPort.WriteLine(GetPRSfromByteArray(CMD));
            // Read simple protocoll string and convert to byte array
            return GetByteArrayfromPRS(TWNPort.ReadLine());
        }// End of DoTXRX

        #region Tools for connect TWN4
        private string RegHKLMQuerySZ(string SubKey, string ValueName)
        {
            string Data;

            RegistryKey Key = Registry.LocalMachine.OpenSubKey(SubKey);
            if (Key == null)
                return "";
            if (Key.GetValue(ValueName) != null)
                Data = Key.GetValue(ValueName).ToString();
            else
                return "";
            if (Data == "")
                return "";
            if (Key.GetValueKind(ValueName) != RegistryValueKind.String)
                Data = "";
            Key.Close();
            return Data;
        }// End of RegHKLMQuerySZ
        private string FindUSBDevice(string Driver, string DevicePath)
        {
            int PortIndex = 0;

            while (true)
            {
                string Path = "SYSTEM\\CurrentControlSet\\Services\\" + Driver + "\\Enum";
                string Data = RegHKLMQuerySZ(Path, PortIndex.ToString());
                PortIndex++;
                if (Data == "")
                    return "";
                string substr = Data.Substring(0, DevicePath.Length).ToUpper();
                if (substr == DevicePath)
                    return Data;
            }
        }// End of FindUSBDevice
        private int GetCOMPortNr(string Device)
        {
            string Path = "SYSTEM\\CurrentControlSet\\Enum\\" + Device + "\\Device Parameters";
            string Data = RegHKLMQuerySZ(Path, "PortName");
            if (Data == "")
                return 0;
            if (Data.Length < 4)
                return 0;
            int PortNr = Convert.ToUInt16(Data.Substring(3));
            if (PortNr < 1 || PortNr > 256)
                return 0;
            return PortNr;
        }// End of GetCOMPortNr
        private string GetCOMPortString(int PortNr)
        {
            if (PortNr < 1 || PortNr > 256)
                return "";
            return "COM" + PortNr.ToString();
        }// End of GetCOMPortString
        private string GetTWNPortName(int PortNr)
        {
            string PortName;
            if (PortNr == 0)
            {
                string Path = FindUSBDevice("usbser", "USB\\VID_09D8&PID_0420\\");

                if (Path == "")
                    return "";
                PortName = GetCOMPortString(GetCOMPortNr(Path));
                if (PortName == "")
                    return "";
            }
            else
            {
                PortName = GetCOMPortString(PortNr);
                if (PortName == "")
                    return "";
            }
            return PortName;
        }// End of GetTWNPortName
        #endregion

        #endregion

        #region Tool for byte arrays
        private byte[] AddByteArray(byte[] Source, byte[] Add)
        {
            // Is Source = null
            if (Source == null)
            {
                // Yes, copy Add in Source
                Source = Add;
                // Return source
                return Source;
            }
            // Initialize buffer array, with the length of Source and Add
            byte[] buffer = new byte[Source.Length + Add.Length];
            // Copy Source in buffer
            for (int i = 0; i < Source.Length; i++)
                // Copy source bytes to buffer
                buffer[i] = Source[i];
            // Add the secound array to buffer
            for (int i = Source.Length; i < buffer.Length; i++)
                // Copy Add bytes after the Source bytes in buffer
                buffer[i] = Add[i - Source.Length];
            // Return the combined array buffer
            return buffer;
        }// End of AddByteArray
        private byte[] AddByte2Array(byte[] Source, byte Add)
        {
            if (Source == null)
            {
                return new byte[] { Add };
            }
            // Initialize buffer with the length of Source + 1
            byte[] buffer = new byte[Source.Length + 1];
            // Copy Source in buffer
            for (int i = 0; i < Source.Length; i++)
                // Copy Source bytes in buffer array
                buffer[i] = Source[i];
            // Add byte behind the Source
            buffer[Source.Length] = Add;
            // Return the buffer
            return buffer;
        }// End of AddByte2Array
        private byte[] GetSegmentFromByteArray(byte[] Source, int index, int count)
        {
            // Initialize buffer with the segment size
            byte[] buffer = new byte[count];
            // Copy bytes from index until count
            for (int i = index; i < (index + count); i++)
                // Copy in segment buffer
                buffer[i - index] = Source[i];
            // Return segment buffer
            return buffer;
        }// End of GetSegmentFromByteArray
        private bool CompareArraysSegments(byte[] Array1, int index1, byte[] Array2, int index2, int count)
        {
            // Plausibility check, is index + count longer than arran
            if (((index1 + count) > Array1.Length) || ((index2 + count) > Array2.Length))
                // Yes, return false
                return false;
            // Compare segments of count
            for (int i = 0; i < count; i++)
                // Is byte in Array1 == byte in Array2?
                if (Array1[i + index1] != Array2[i + index2])
                    // No, return flase
                    return false;
            // Return true
            return true;
        }// End of CompareArraysSegment
        #endregion
    }

    class TSNEPCMD
    {
        public string Name;
        public int Token;
        public TSNEPCMD(string Name, int Token)
        {
            this.Name = Name;
            this.Token = Token;
        }
        public override string ToString()
        {
            return Name;
        }

    }

    class TURITypes : List<TSNEPCMD>
    {
        public TURITypes()
            : base()
        {
            Add(new TSNEPCMD("", 0));
            Add(new TSNEPCMD("http://www.", 1));
            Add(new TSNEPCMD("https://www.", 2));
            Add(new TSNEPCMD("http://", 3));
            Add(new TSNEPCMD("https://", 4));
            Add(new TSNEPCMD("tel:", 5));
            Add(new TSNEPCMD("mailto:", 6));
            Add(new TSNEPCMD("ftp://anonoymous:anonymous", 7));
            Add(new TSNEPCMD("ftp://ftp", 8));
            Add(new TSNEPCMD("ftps://", 9));
            Add(new TSNEPCMD("sftp://", 10));
            Add(new TSNEPCMD("smb://", 11));
            Add(new TSNEPCMD("nfs://", 12));
            Add(new TSNEPCMD("ftp://", 13));
            Add(new TSNEPCMD("dav://", 14));
            Add(new TSNEPCMD("news:", 15));
            Add(new TSNEPCMD("telnet://", 16));
            Add(new TSNEPCMD("imap:", 17));
            Add(new TSNEPCMD("rtsp://", 18));
            Add(new TSNEPCMD("urn:", 19));
            Add(new TSNEPCMD("pop:", 20));
            Add(new TSNEPCMD("sip:", 21));
            Add(new TSNEPCMD("sips:", 22));
            Add(new TSNEPCMD("tftp:", 23));
            Add(new TSNEPCMD("btspp://", 24));
            Add(new TSNEPCMD("btl2cap://", 25));
            Add(new TSNEPCMD("btgoep://", 26));
            Add(new TSNEPCMD("tcpobex://", 27));
            Add(new TSNEPCMD("irdaobex://", 28));
            Add(new TSNEPCMD("file://", 29));
            Add(new TSNEPCMD("urn:epc:id:", 30));
            Add(new TSNEPCMD("urn:epc:tag:", 31));
            Add(new TSNEPCMD("urn:epc:pat:", 32));
            Add(new TSNEPCMD("urn:epc:raw:", 33));
            Add(new TSNEPCMD("urn:epc:", 34));
            Add(new TSNEPCMD("urn:nfc:", 35));
        }
    }

}
